using System;
using System.Reflection;
using System.Xml;
using System.Text;
using System.Configuration;
using System.Collections.Specialized;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.AppServices;
using gov.va.med.vbecs.Common.Log;


namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// <see cref="GlobalConfig"/> class provides configuration settings for the 
	/// VistALink Open Library. It automatically detects if VistALink section exists 
	/// in the calling application configuration file, if yes - it takes all 
	/// available settings from the latter. If not - default hardcoded values are
	/// used. The class, however, does not provide a standard way to modify its
	/// properties programmatically. 
	/// </summary>
	public class GlobalConfig
	{
		// Consts used to read settings from config file
		private const string XMLCONSTS_CONFIG_ROOT_SECTION_NAME = "VistALink";
		
		// Variables containing configured values
		private static Encoding _networkMessageEncoding;
		private static int _maxRpcParametersNumber;
		private static string _rpcXmlNullValue;
		private static byte _eotByte;
		private static Timeout _defaultRpcRequestTimeout;
		private static ServerConnectionInfoCollection _serverList;

	    private static bool _configurationFromDBAcquired;
        private static readonly object Lock = new object();

        // logger object
        private static ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        
		/// <summary>
		/// Class should not be instantiated.
		/// </summary>
		private GlobalConfig() {}

		/// <summary>
        /// Initializes <see cref="GlobalConfig"/> on first request. Acquired
		/// </summary>
		static GlobalConfig()
		{
            _logger.Debug("VistA -> GlobalConfig called");
			SetDefaultHardcodedValues();
		    _configurationFromDBAcquired = false;
			
			GetServerListFromConfigFile();
		}

	    static private void EnsureInitialized()
	    {
            _logger.Debug("VistA -> EnsureInitialized");
	        lock (Lock)
	        {
	            if (_configurationFromDBAcquired)
	                return;

	            // Override default configuration values here
	            // CR 3304
	            try
	            {
	                _defaultRpcRequestTimeout =
	                    new Timeout(
	                        (int) GlobalContext.Instance().AppSettingsReader.GetValue("VistAClientTimeout", typeof (int))*
	                        1000);
	                _logger.Debug(string.Format("VistAClientTimeout configuration is {1} seconds for application{0}",
	                    GlobalContext.Instance().AppSettingsReader.AppName, _defaultRpcRequestTimeout.ToInt32Seconds()));
	            }
	            catch (ArgumentException e)
	            {
	                _logger.Warn(
	                    string.Format(
	                        "No VistAClientTimeout configuration is not found for {1} application{0}Use default value {2} seconds{0}Error message: {3}",
	                        Environment.NewLine, GlobalContext.Instance().AppSettingsReader.AppName,
	                        _defaultRpcRequestTimeout.ToInt32Seconds(), e.Message));
	            }
	            catch (ConfigurationErrorsException e)
	            {
	                _logger.Warn(
	                    string.Format(
	                        "No some configuration is not found{0}Use default value {1} seconds{0}Error message: {2}",
	                        Environment.NewLine, _defaultRpcRequestTimeout.ToInt32Seconds(), e.Message));
	            }
	            catch (SqlConnectionException e)
	            {
	                _logger.Warn(
	                    string.Format(
	                        "No database connection failed{0}Use default value {1} seconds{0}Error message: {2}",
	                        Environment.NewLine, _defaultRpcRequestTimeout.ToInt32Seconds(), e.Message));
	            }

	            //TODO: override other values here?

	            _configurationFromDBAcquired = true;
	        }
	    }

	    /// <summary>
		/// Defines encoding for VistALink protocol network messages
		/// </summary>
		public static Encoding NetworkMessageEncoding
		{
			get
			{
				return _networkMessageEncoding;
			}
		}

		/// <summary>
		/// Maximum number of RPC request parameters 
		/// specified in VistALink protocol.
		/// </summary>
		public static int MaxRpcParametersNumber 
		{
			get
			{
				return _maxRpcParametersNumber;
			}
		}

		/// <summary>
		/// This value is used to substitute null values when forming 
		/// RPC request to M server. 
		/// </summary>
		public static string RpcXmlNullValue
		{
			get
			{
				return _rpcXmlNullValue;
			}
		}

		/// <summary>
		/// End-of-transmission (EOT) character used to separate network messages.
		/// When defined in config file, integer code must be used. 
		/// </summary>
		public static byte EOTByte
		{
			get
			{
				return _eotByte;
			}
		}

		/// <summary>
		/// Default RPC timeout. 
		/// </summary>
		public static Timeout DefaultRpcRequestTimeout
		{
			get
			{
                // this property requires initialization 
			    EnsureInitialized();
				return _defaultRpcRequestTimeout;
			}
		}

		/// <summary>
		/// Sets default hardcoded values. 
		/// </summary>
		private static void SetDefaultHardcodedValues()
		{
			_networkMessageEncoding = Encoding.ASCII;
			_maxRpcParametersNumber = 10;
			_rpcXmlNullValue = String.Empty;
			_eotByte = Convert.ToByte( 4 );
			_defaultRpcRequestTimeout = new Timeout( 30000 );
			_serverList = null;
		}
		
		/// <summary>
		/// Retrieves list of servers from application configuration file.
		/// </summary>
		private static void GetServerListFromConfigFile()
		{
            _serverList = 
				(ServerConnectionInfoCollection)System.Configuration.ConfigurationManager.GetSection( 
					GetXPathToChildConfigSection( ServerListConfigSectionHandler.SectionRootElementName ) );					
		}

		/// <summary>
		/// The method provides a standard way to store and access M server connection
		/// information in application configuration file. It will return server info,
		/// if the record for the server with given handle is available, and null otherwise. 
		/// Please, understand, that it will only work when server information
		/// is available in mapped configuration file. 
		/// </summary>
		/// <param name="serverHandle">
		///		Internal handle used to indentify M server in configuration file
		///	</param>
		/// <returns>
		///		Instance of <see cref="ServerConnectionInfo"/> if server information is found. 
		///		Null otherwise.
		///	</returns>
		public static ServerConnectionInfo GetServerConnectionInfoByHandle( string serverHandle )
		{
			if( serverHandle == null )
				throw( new ArgumentNullException( "serverHandle" ) );

			if( _serverList == null ) 
				return null;
			
			return _serverList.GetServerConnectionInfoByHandle( serverHandle );
		}

		/// <summary>
		/// Forms string XPath to a section with a given name that 
		/// is a child of VistALink application configuration section.
		/// </summary>
		/// <param name="childSectionName">Name of child section to form XPath to.</param>
		/// <returns>
		///		String XPath to a given child section of 
		///		VistALink application configuration section.
		///	</returns>
		public static string GetXPathToChildConfigSection( string childSectionName )
		{
			return String.Format( "{0}/{1}", XMLCONSTS_CONFIG_ROOT_SECTION_NAME, childSectionName ); 
		}
	}
}
